local super = require "TableColumn"

DateColumn = super:new()

local defaults = {
    halign = 0,
    valign = 0.5,
}

local nilDefaults = {
    'value', 'formatter', 'paint', 'font',
}

local getterNames = {'value', 'formatter', 'paint', 'font', 'halign', 'valign'}

local inspectorInfo = {
    {'KeyArtifact', {'value'}, 'Value'},
}
local moreInspectorInfo = {
    {'Color', {'getPaint:setPaint', custom = 'hasExplicitPaint:'}, 'Color'},
    {'Font', {'getFont:setFont', custom = 'hasExplicitFont:'}, 'Font'},
    {'Alignment', {'halign'}, 'Alignment'},
}

function DateColumn:new()
    self = super.new(self)
    
    for k, v in pairs(defaults) do
        self:addProperty(k, v)
    end
    for _, k in pairs(nilDefaults) do
        self:addProperty(k)
    end
    
    self.stamp = FormattedValueTextStamp
    self.getterNames = getterNames
    self.inspectorInfo = inspectorInfo
    
    local validator = function(formatter)
        return formatter == nil or Object.isa(formatter, DateFormatter)
    end
    self:getPropertyHook('formatter'):setValidator(validator)
    
    return self
end

function DateColumn:unarchived()
    local dataset = self:getDataset()
    if dataset then
        if self:getProperty('value') == nil then
            local avoidingFields = self:peerPropertyKeyArtifactValues(DateColumn, 'value')
            local field = dataset:pickField('Date', avoidingFields)
            if field then
                self:setProperty('value', KeyArtifact:new(field))
                self:setTitle(field)
            end
        end
    end
    super.unarchived(self)
end

function DateColumn:getInspectors()
    local list = super.getInspectors(self)
    local inspector, hook
    inspector = Inspector:new{
        title = 'Format',
        type = 'Class',
        constraint = function()
            return self:getFormatters()
        end,
    }
    hook = Hook:new(
        function()
            local formatter = self:getFormatter()
            if Object.isa(formatter, DateFormatter) then
                return formatter:getTemplate()
            end
        end,
        function(value)
            local formatter
            if type(value) == 'string' then
                formatter = DateFormatter:new(value)
            end
            self:setFormatter(formatter)
        end)
    inspector:addHook(hook)
    self:getPropertyHook('formatter'):addObserver(inspector)
    list:add(inspector)
    for i = 1, #moreInspectorInfo do
        local info = moreInspectorInfo[i]
        list:add(self:createInspector(unpack(info)))
    end
    return list
end

function DateColumn:getFormatter()
    return self:getProperty('formatter')
end

function DateColumn:setFormatter(formatter)
    self:setProperty('formatter', formatter)
end

function DateColumn:getFormatters()
    local formatters = { { nil, 'Auto Format' } }
    appendtables(formatters, {false}, DateFormatter:getTemplates())
    return formatters
end

function DateColumn:getFont()
    return self:getProperty('font') or self:getParent():getLabelFont()
end

function DateColumn:setFont(value)
    self:setProperty('font', value)
end

function DateColumn:hasExplicitFont()
    return self:getProperty('font') ~= nil
end

return DateColumn
